import {
    View,
    Text,
    ScrollView,
    TouchableOpacity,
    Dimensions,
    Pressable,
} from "react-native";
import React, { useEffect, useState } from "react";
import { useSelector, useDispatch } from "react-redux";
import CommonStyles from "../../Utilities/CommonStyles/CommonStyles";
import { orderSummaryStyle } from "../OrderSummary/OrderSummaryStyle";
import { OrderIndicator } from "../OrderSummary/OrderSummary";
import SubtractIcon from "../../../assets/svgs/cart/subtract.svg";
import { orderConfirmedStyle } from "./OrderConfirmedStyle";
import { OrderDetailsStyle } from "../../Profile/OrderHistory/OrderDetails/OrderDetailsStyle";
import dpr from "../../Utilities/CustomStyleAttribute/dpr";
import { EditProfileStyle } from "../../Profile/EditProfile/StyleEditProfile";
import CustomSpinner from "../../Utilities/CustomSpinner/CustomSpinner";
import { addNewAddressStyle } from "../../Profile/Address/AddNewAddress/AddNewAddressStyle";
import { getOrderDetails } from "../../../redux/slices/user/orderDetails/orderDetails";
import useAuth from "../../../hooks/useAuth";
import OrderDetailsSkeleton from "../../../src/skeletons/screens/profile/order/OrderDetails/OrderDetailsSkeleton";
import useProcessPrice from "../../../hooks/useProcessPrice";
import { BASE_API_URL_STG } from "@env";
import useLangTranslation from "../../../hooks/useLangTranslation";
import useGuestUser from "../../../hooks/useGuestUser";
import gapStyle from "../../Utilities/CommonStyles/gap.style";
import { PRODUCT_DETAILS } from "../../../components/Navigation/RouteNames";
import CustomSVG from "../../../src/components/CustomSVG/CustomSVG";
import { splitCamelCase } from "../../../helper/splitCamelCase";

const OrderConfirmed = (props) => {
    const isGuest = useGuestUser();
    const dispatch = useDispatch();
    const { trans } = useLangTranslation();
    const { access_token } = useAuth();
    const processPrice = useProcessPrice();
    const [inFileLoading, setInFileLoading] = useState(true);
    const { itemId } = props?.route?.params || {};
    const { orderDetails, loading } = useSelector(
        (state) => state.getOrderDetailsSlice || {}
    );
    const navigateToHome = props.route?.params?.resetToHome

    const {
        created_at = "",
        shipping,
        payment,
        line_items,
        total,
        payment_status,
        shipping_title,
        order_key,
    } = orderDetails || {};

    const [estimateDelivery, setEstimateDelivery] = useState(null);

    useEffect(() => {
        let isMounted = true;
        if (isMounted) {
            const URL = `${BASE_API_URL_STG}/user/orders/${itemId}`;
            if (isGuest) {
                dispatch(getOrderDetails({ URL }));
            } else {
                dispatch(getOrderDetails({ access_token, URL }));
            }
            setInFileLoading(false);
        }
        return () => {
            isMounted = false;
        };
    }, []);

    useEffect(() => {
        const calculateEstimateDelivery = (orders) => {
            let filteredOrders = orders?.filter(
                (order) => order.estimate_delivery !== null
            );
            if (filteredOrders.length === 0) return null;
            let min = filteredOrders[0].estimate_delivery;
            let max = filteredOrders[0].estimate_delivery;
            for (let i = 1; i < filteredOrders.length; i++) {
                if (filteredOrders[i].estimate_delivery < min) {
                    min = filteredOrders[i].estimate_delivery;
                }
                if (filteredOrders[i].estimate_delivery > max) {
                    max = filteredOrders[i].estimate_delivery;
                }
            }
            if (min === max) {
                return min;
            }
            return `${min} - ${max}`;
        };
        if (line_items?.length) {
            const result = calculateEstimateDelivery(line_items);
            setEstimateDelivery(result);
        }
    }, [line_items?.length]);

    useEffect(() => {
        if (navigateToHome) {
            props.navigation.addListener('beforeRemove', e => {
            e.preventDefault();
            props.navigation.dispatch(props.navigation.navigate("Home"));
            });
        }
    }, [props.navigation, navigateToHome]);

    return (
        <>
            <View style={CommonStyles.container}>
                <View style={CommonStyles.customHeaderContainer}>
                    <Text style={CommonStyles.headerName}>
                        {props.route.name.toUpperCase()}
                    </Text>
                </View>
            </View>
            {loading || inFileLoading ? (
                <OrderDetailsSkeleton />
            ) : (
                <>
                    <ScrollView showsVerticalScrollIndicator={false}>
                        <View style={CommonStyles.globalContainer}>
                            <View style={orderSummaryStyle.indicatorCont}>
                                {[
                                    "#33C172",
                                    "#33C172",
                                    "#33C172",
                                    "#33C172",
                                ].map((item, i) => (
                                    <OrderIndicator key={`key${i}`} bg={item} />
                                ))}
                            </View>
                            <View style={orderConfirmedStyle.topPosition}>
                                <View style={orderConfirmedStyle.iconAlign}>
                                    <Text style={[orderConfirmedStyle.topIcon]}>
                                        <CustomSVG svgIcon={SubtractIcon} />
                                    </Text>
                                </View>
                                <Text style={orderConfirmedStyle.topText1}>
                                    {trans("Thank you")}
                                </Text>
                                <Text style={orderConfirmedStyle.topText2}>
                                    {trans("Your order has been received.")}
                                </Text>
                            </View>
                            <View
                                style={[
                                    OrderDetailsStyle.deliveryCont,
                                    OrderDetailsStyle.invoiceCont,
                                ]}
                            >
                                <View style={OrderDetailsStyle.deliverySubCont}>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Invoice Number")}
                                    </Text>
                                    <Text
                                        style={
                                            orderConfirmedStyle.deliveryText2
                                        }
                                    >
                                        {payment?.code || order_key}
                                    </Text>
                                </View>
                                <View
                                    style={[
                                        OrderDetailsStyle.deliverySubCont,
                                        OrderDetailsStyle.borderStyle
                                    ]}
                                >
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Grand Total")}
                                    </Text>
                                    <Text
                                        style={
                                            orderConfirmedStyle.deliveryText2
                                        }
                                    >
                                        {processPrice(total)}
                                    </Text>
                                </View>
                            </View>
                            <View style={OrderDetailsStyle.deliveryCont}>
                                <View style={OrderDetailsStyle.deliverySubCont}>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Shipping Method")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {shipping_title && trans(shipping_title)}
                                    </Text>
                                </View>
                                <View
                                    style={[
                                        OrderDetailsStyle.deliverySubCont,
                                        OrderDetailsStyle.borderStyle
                                    ]}
                                >
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Payment Status")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {payment_status && trans(payment_status)}
                                    </Text>
                                </View>
                            </View>
                            <View
                                style={[
                                    OrderDetailsStyle.deliveryCont,
                                    gapStyle.mt15
                                ]}
                            >
                                <View style={OrderDetailsStyle.deliverySubCont}>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText1}
                                    >
                                        {trans("Order Date")}
                                    </Text>
                                    <Text
                                        style={OrderDetailsStyle.deliveryText2}
                                    >
                                        {created_at}
                                    </Text>
                                </View>
                                {estimateDelivery != null ? (
                                    <View
                                        style={[
                                            OrderDetailsStyle.deliverySubCont,
                                            OrderDetailsStyle.deliverySubContainer
                                        ]}
                                    >
                                        <Text
                                            style={
                                                OrderDetailsStyle.deliveryText1
                                            }
                                        >
                                            {trans("Estimated Delivery")}
                                        </Text>
                                        <Text
                                            style={
                                                OrderDetailsStyle.deliveryText2
                                            }
                                        >
                                            {trans("{{x}} days (all items)", {x: estimateDelivery})}
                                        </Text>
                                    </View>
                                ) : payment?.gateway ?
                                    <View style={[
                                        OrderDetailsStyle.deliverySubCont,
                                        OrderDetailsStyle.deliverySubContainer
                                    ]}>
                                        <Text
                                            style={[OrderDetailsStyle.deliveryText1]}
                                        >
                                            {trans("Payment Method")}
                                        </Text>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText2}
                                        >
                                            {splitCamelCase(payment?.gateway)}
                                        </Text>
                                    </View>
                                    : null
                                }
                            </View>
                            {(estimateDelivery != null && payment?.gateway) &&
                                <View
                                    style={[
                                        OrderDetailsStyle.deliveryCont,
                                        gapStyle.mt15
                                    ]}
                                >
                                    <View style={OrderDetailsStyle.deliverySubCont}>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText1}
                                        >
                                            {trans("Payment Method")}
                                        </Text>
                                        <Text
                                            style={OrderDetailsStyle.deliveryText2}
                                        >
                                            {splitCamelCase(payment?.gateway)}
                                        </Text>
                                    </View>
                                </View>
                            }
                            <View style={OrderDetailsStyle.orderProductsCont}>
                                <Text
                                    style={OrderDetailsStyle.orderProductTitle}
                                >
                                    {trans("Ordered Products")}
                                </Text>
                                {line_items?.map((item, index) => (
                                    <View key={`key` + index}>
                                        <Pressable
                                            onPress={() => {
                                                props.navigation.navigate(
                                                    PRODUCT_DETAILS,
                                                    {
                                                        slug: item?.slug,
                                                    }
                                                );
                                            }}
                                            style={[
                                                OrderDetailsStyle.singleProductCont,
                                                OrderDetailsStyle.lastProductCont(index, line_items?.length)
                                            ]}
                                        >

                                            <View>
                                                <Text
                                                    style={
                                                        OrderDetailsStyle.name
                                                    }
                                                >
                                                    {item?.name.slice(0, 20)} .
                                                    . .
                                                </Text>
                                                <View
                                                    style={
                                                        OrderDetailsStyle.items
                                                    }
                                                >
                                                    <Text
                                                        style={
                                                            OrderDetailsStyle.quantity
                                                        }
                                                    >
                                                        {item?.quantity} x {trans("Item")}
                                                    </Text>
                                                    <View
                                                        style={
                                                            OrderDetailsStyle.vrLine
                                                        }
                                                    />
                                                    <Text
                                                        style={
                                                            OrderDetailsStyle.quantity
                                                        }
                                                    >
                                                        {
                                                            item?.vendor?.shop
                                                                ?.name
                                                        }
                                                    </Text>
                                                </View>
                                            </View>
                                            <View>
                                                <Text
                                                    style={
                                                        OrderDetailsStyle.name
                                                    }
                                                >
                                                    {processPrice(
                                                        parseInt(item?.price) *
                                                            parseInt(
                                                                item?.quantity
                                                            )
                                                    )}
                                                </Text>
                                            </View>
                                        </Pressable>
                                    </View>
                                ))}
                            </View>
                        </View>
                    </ScrollView>
                    <View style={EditProfileStyle.changeInfo}>
                        <TouchableOpacity
                            onPress={() =>
                                props.navigation.push("order details", {
                                    itemId,
                                    resetToHome: true,
                                })
                            }
                        >
                            {!isGuest && (
                                <Text style={EditProfileStyle.cancel}>
                                    {trans("View Details")}
                                </Text>
                            )}
                        </TouchableOpacity>
                        {false ? (
                            <View style={orderConfirmedStyle.customSpinnerWrapper}>
                                <CustomSpinner
                                    filePath={require("../../../assets/lottie/loader2.json")}
                                    size={{
                                        width: dpr(60),
                                        height: dpr(50),
                                    }}
                                />
                            </View>
                        ) : (
                            <TouchableOpacity
                                onPress={() =>
                                    props.navigation.navigate("Home")
                                }
                                style={addNewAddressStyle.saveAddressCont}
                            >
                                <Text style={addNewAddressStyle.saveAddressBtn}>
                                    {trans("Back to Home")}
                                </Text>
                            </TouchableOpacity>
                        )}
                    </View>
                </>
            )}
        </>
    );
};

export default OrderConfirmed;
